package com.tencent.start.cgs.tools;

import org.apache.commons.codec.binary.Hex;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.nio.file.PathMatcher;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import static com.tencent.start.cgs.tools.ZpkFile.*;
import static com.tencent.start.cgs.tools.ZpkMetadata.FileNode;

public class ZpkFilePack extends App {
    protected RandomAccessOutputFile output;
    protected final long buildTime = System.currentTimeMillis();
    protected final CRC32 crc32 = new CRC32();
    protected final MessageDigest md5;
    protected final Map<String, FileNode> fileNodesMap = new HashMap<>(128 * 1024);
    protected static final int AWB_FLAG_NONE = 0;
    protected static final int AWB_FLAG_FIRST = 1;
    protected static final int AWB_FLAG_LAST = 2;

    protected interface FileDataWriter {
        void write(FileNode fileNode, AbstractFile file) throws IOException;
        default void finish() throws IOException {}
    }

    protected static class AsyncWriteUserData {
        public int crc = 0;
        public FileNode exists = null;
        public long pos = 0;
    }

    protected static class AsyncWriteBuffer {
        final FileNode fileNode;
        final byte[] buf;
        final int off;
        final int len;
        final int flag;

        AsyncWriteBuffer(FileNode fileNode, byte[] buf, int off, int len, int flag) {
            this.fileNode = fileNode;
            this.buf = buf;
            this.off = off;
            this.len = len;
            this.flag = flag;
        }

        AsyncWriteBuffer(FileNode fileNode, int flag) {
            this(fileNode, null, 0, 0, flag);
        }
    }

    protected void onWriteFileBegin(FileNode fileNode) throws IOException {}
    protected void onWriteFile(FileNode fileNode, byte[] buf, int off, int len) throws IOException {
        output.write(buf, off, len);
    }
    protected void onWriteFileEnd(FileNode fileNode, boolean canceled) throws IOException {}

    protected final static AsyncWriteBuffer EXIT_FLAGS = new AsyncWriteBuffer(null, -1);

    @SuppressWarnings("unused")
    protected class AsyncFileDataWriter implements FileDataWriter, Runnable {
        Thread asyncThread;
        protected final BlockingQueue<AsyncWriteBuffer> asyncQueue = new LinkedBlockingQueue<>(8);
        protected final BlockingQueue<AsyncWriteBuffer> writeQueue = new LinkedBlockingQueue<>(8);

        protected void calcWork() throws Exception {
            AsyncWriteBuffer awb;
            while (EXIT_FLAGS != (awb = asyncQueue.take())) {
                final FileNode fileNode = awb.fileNode;
                final AsyncWriteUserData ud = (AsyncWriteUserData) fileNode.ud;
                if (AWB_FLAG_FIRST == awb.flag) {
                    md5.reset();
                    ud.crc = crc32.crc();
                }
                if (awb.len > 0) {
                    crc32.update(awb.buf, awb.off, awb.len);
                    md5.update(awb.buf, awb.off, awb.len);
                }
                if (AWB_FLAG_LAST == awb.flag) {
                    fileNode.md5sum = md5.digest();
                    if (null != (ud.exists = putIfAbsent(fileNode))) {
                        crc32.reset(ud.crc);
                    }
                }
                writeQueue.put(awb);
            }
        }

        protected void writeWork() throws Exception {
            AsyncWriteBuffer awb;
            while (EXIT_FLAGS != (awb = writeQueue.take())) {
                final FileNode fileNode = awb.fileNode;
                final AsyncWriteUserData ud = (AsyncWriteUserData) fileNode.ud;
                if (AWB_FLAG_FIRST == awb.flag) {
                    ud.pos = output.getFilePointer();
                    fileNode.dataOffset = getDataOffset();
                    onWriteFileBegin(fileNode);
                }
                if (awb.len > 0 && null == ud.exists) {
                    onWriteFile(fileNode, awb.buf, awb.off, awb.len);
                }
                if (AWB_FLAG_LAST == awb.flag) {
                    if (null != ud.exists) {
                        output.seek(ud.pos);
                        fileNode.dataOffset = ud.exists.dataOffset;
                        fileNode.md5sum = ud.exists.md5sum;
                        fileNode.ud = null;
                        traceTheSame(fileNode, ud.exists, true);
                    }
                    onWriteFileEnd(fileNode, null != ud.exists);
                }
            }
        }

        @Override
        public void run() {
            Thread thread = new Thread(() -> {
                try {
                    writeWork();
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            thread.start();
            try {
                calcWork();
                writeQueue.put(EXIT_FLAGS);
                thread.join();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        AsyncFileDataWriter() {
            (asyncThread = new Thread(this)).start();
        }

        @SuppressWarnings("UnusedReturnValue")
        @Override
        public void write(FileNode fileNode, AbstractFile file) throws IOException {
            if (file.length() != fileNode.fileSize) {
                throw new IOException("file length");
            }
            fileNode.ud = new AsyncWriteUserData();
            try {
                asyncQueue.put(new AsyncWriteBuffer(fileNode, AWB_FLAG_FIRST));
                readFileWithProgress(file, (buf, off, len) -> {
                    try {
                        asyncQueue.put(new AsyncWriteBuffer(fileNode, buf, off, len, AWB_FLAG_NONE));
                    } catch (InterruptedException e) {
                        throw new IOException(e);
                    }
                    return null;
                });
                asyncQueue.put(new AsyncWriteBuffer(fileNode, AWB_FLAG_LAST));
            } catch (InterruptedException e) {
                throw new IOException(e);
            }
        }

        @Override
        public void finish() throws IOException {
            try {
                asyncQueue.put(EXIT_FLAGS);
                asyncThread.join();
            } catch (InterruptedException e) {
                throw new IOException(e);
            }
        }
    } // class AsyncFileDataWriter

    @SuppressWarnings("unused")
    protected class SyncFileDataWriter implements FileDataWriter {
        @Override
        public void write(FileNode f, AbstractFile file) throws IOException {
            md5.reset();
            final int crc = crc32.crc();
            final FileNode fileNode = f;
            fileNode.dataOffset = getDataOffset();
            onWriteFileBegin(fileNode);
            readFileWithProgress(file, (buf, off, len) -> {
                crc32.update(buf, off, len);
                md5.update(buf, off, len);
                onWriteFile(fileNode, buf, off, len);
                return buf;
            });
            fileNode.md5sum = md5.digest();
            FileNode exists = putIfAbsent(fileNode);
            if (null != exists) {
                crc32.reset(crc);
                output.seek(fileNode.dataOffset);
                fileNode.dataOffset = exists.dataOffset;
                fileNode.md5sum = exists.md5sum;
                traceTheSame(fileNode, exists, true);
            }
            onWriteFileEnd(fileNode, null != exists);
        }
    } // class SyncFileDataWriter

    final FileDataWriter fileDataWriter = new AsyncFileDataWriter();
    //final FileDataWriter fileDataWriter = new SyncFileDataWriter();

    public ZpkFilePack() throws IOException {
        try {
            md5 = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
    }

    protected void traceTheSame(FileNode fileNode, FileNode exists, boolean strict) {
        App.Assert(fileNode.fileSize == exists.fileSize);
        System.out.println("File exists" + (strict ? "" : " (not strict)") + ": \"" +
                fileNode.relativeName + "\" (" + sdf1.format(fileNode.lastModified) + ") -> \"" +
                exists.relativeName + "\" (" + sdf1.format(exists.lastModified) + "), size=" +
                humanReadableBytes(fileNode.fileSize) + ", md5=" + Hex.encodeHexString(exists.md5sum));
    }

    protected FileNode putIfAbsent(FileNode fileNode) {
        return (fileNode.isDirectory() || 0 == fileNode.fileSize) ? null : fileNodesMap.putIfAbsent(
                Hex.encodeHexString(fileNode.md5sum) + fileNode.fileSize, fileNode);
    }

    protected long getDataOffset() throws IOException {
        return output.getFilePointer();
    }

    protected void onEnumerationBegin() throws IOException {}

    protected void onEnumerationEnd() throws IOException {}

    public static class DataSectionInfo {
        public long usedBytes;
        public long metadataOffset;
        public int metadataSize;
        public int metadataTotalSize;
        public long dataEndPosition;
    }

    @SuppressWarnings("unused")
    protected void writeFileData(FileNode fileNode, AbstractFile file) throws IOException {
        fileDataWriter.write(fileNode, file);
    }

    protected DataSectionInfo writeDataSection(AbstractFile inputDir, ZpkMetadata.EnumFilter ef) throws IOException {
        final DataSectionInfo result = new DataSectionInfo();
        final String dataEntryName = getTimestampFileName(DATA_FILENAME, buildTime, "");
        final long dataEntryPosition = output.getFilePointer();
        final long dataEntrySize = ZIP_ENTRY_SIZE_64 + dataEntryName.getBytes(StandardCharsets.UTF_8).length;
        final long dataPosition = dataEntryPosition + dataEntrySize;

        crc32.reset();

        // appending tag
        output.seek(dataEntryPosition);
        output.write(getTimestampFileName(ZpkFile.APPEND_TAG, buildTime, "").getBytes());

        // file data
        output.seek(dataPosition);

        onEnumerationBegin();
        FileNode rootNode = ZpkMetadata.enumerate(inputDir, (fileNode, file) -> {
            if (!fileNode.isDirectory() && 0 != fileNode.fileSize) {
                writeFileData(fileNode, file);
            }
        }, ef);
        onEnumerationEnd();

        fileDataWriter.finish();

        result.usedBytes = rootNode.fileSize;
        result.metadataOffset = output.getFilePointer();

        // metadata
        RandomAccessByteArrayOutputStream out = new RandomAccessByteArrayOutputStream(65536 * 1024);
        final ZpkMetadata.BuildResult info = new ZpkMetadata.Builder().build(out, rootNode);
        byte[] fileMetadata = out.toByteArray();
        if (App.ENABLE_DEBUG) {
            ZpkMetadata.check(fileMetadata);
        }
        output.write(fileMetadata, 0, fileMetadata.length);
        crc32.update(fileMetadata, 0, fileMetadata.length);
        result.metadataSize = info.checksumsOffset; // without checksums
        result.metadataTotalSize = fileMetadata.length;

        // footer
        final int footerSize = 32;
        ByteBuffer buff = ByteBuffer.allocate(footerSize + 1);
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.put(ZpkFile.DATA_FOOTER_MAGIC.getBytes());
        buff.putLong(result.metadataOffset - dataPosition);
        buff.putInt(result.metadataSize);
        buff.putInt(result.metadataTotalSize);
        buff.position(footerSize);
        buff.put((byte) footerSize);

        final byte[] footerData = buff.array();
        output.write(footerData, 0, footerData.length);
        crc32.update(footerData, 0, footerData.length);

        final long dataEndPosition = output.getFilePointer();
        final long dataSize = dataEndPosition - dataPosition;
        App.Assert(dataSize == result.metadataOffset - dataPosition + fileMetadata.length + footerSize + 1);
        output.seek(dataEntryPosition);
        ZpkFile.writeEntry(output, dataEntryName, dataSize, buildTime, crc32.getValue(), true);
        App.Assert(output.getFilePointer() == dataPosition);
        output.setLength(dataEndPosition);
        result.dataEndPosition = dataEndPosition;
        return result;
    }

    protected long build(AbstractFile inputDir, ZpkMetadata.EnumFilter ef) throws IOException {
        final int headerSize = 32;
        final long headerEntryPos = output.getFilePointer();
        final long headerEntrySize = ZIP_ENTRY_SIZE + HEADER_FILENAME.getBytes(StandardCharsets.UTF_8).length;
        final long headerPos = headerEntryPos + headerEntrySize;

        output.seek(headerPos + headerSize);
        final DataSectionInfo info = writeDataSection(inputDir, ef);

        // header
        ByteBuffer buff = ByteBuffer.allocate(headerSize);
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.putLong(info.usedBytes);        // UsedBytes
        buff.putLong(info.metadataOffset);   // metadata offset
        buff.putInt(info.metadataSize);      // metadata size
        buff.putInt(info.metadataTotalSize); // metadata total size (with md5)
        buff.putLong(0);               // Reserved 1-64bit

        final byte[] headerData = buff.array();
        crc32.reset();
        crc32.update(headerData, 0, headerSize);
        App.Assert(headerData.length == headerSize);
        output.seek(headerEntryPos);
        ZpkFile.writeEntry(output, HEADER_FILENAME, headerSize, buildTime, crc32.getValue(), false);
        App.Assert(headerPos == output.getFilePointer());
        output.write(headerData, 0, headerSize);
        App.Assert(headerPos + headerSize == output.getFilePointer());
        return info.dataEndPosition - headerEntryPos;
    }

    public static AbstractFile getTempFile(AbstractFile outputFile) throws IOException {
        AbstractFile tempFile = outputFile.getParentFile().getChildFile(
                "~" + App.randomString(7) + "." + outputFile.getName() + ".tmp");
        if (!tempFile.getParentFile().mkdirs() && tempFile.exists() && !tempFile.delete()) {
            throw new IOException("count not delete file: " + tempFile);
        }
        return tempFile;
    }

    public long build(AbstractFile outputFile, AbstractFile inputFile, ZpkMetadata.EnumFilter ef) throws IOException {
        AbstractFile tempFile = getTempFile(outputFile);
        try {
            return build(tempFile, outputFile, inputFile, ef);
        } catch (Exception e) {
            tempFile.delete();
            throw new IOException(e);
        }
    }

    protected long build(AbstractFile tempFile, AbstractFile outputFile, AbstractFile inputFile,
                         ZpkMetadata.EnumFilter ef) throws IOException {
        long fileSize;
        try (RandomAccessOutputFile out = RandomAccessOutputFile.from(tempFile)) {
            output = out;
            output.order(ByteOrder.LITTLE_ENDIAN);
            fileSize = build(inputFile, ef);
            output.flush();
            output = null;
        }
        if (outputFile.exists() && !outputFile.delete()) {
            throw new IOException("count not delete file: " + outputFile);
        } else if (!tempFile.renameTo(outputFile)) {
            throw new IOException("count not rename: " + outputFile);
        }
        return fileSize;
    }

    public static class ExcludesEnumFilter implements ZpkMetadata.EnumFilter {

        public List<PathMatcher> matchers;

        public ExcludesEnumFilter(List<String> patterns) {
            if (patterns != null) {
                this.matchers = new ArrayList<>(patterns.size());
                java.nio.file.FileSystem fs = java.nio.file.FileSystems.getDefault();
                for (String pattern : patterns) {
                    this.matchers.add(fs.getPathMatcher("glob:" + pattern));
                }
            } else {
                this.matchers = new ArrayList<>();
            }
        }

        @Override
        public boolean isIgnored(String path) {
            for (java.nio.file.PathMatcher matcher : this.matchers) {
                if (matcher.matches(java.nio.file.Path.of(path))) {
                    return true;
                }
            }
            return false;
        }
    }

    public static class IncludesEnumFilter extends ExcludesEnumFilter {
        public IncludesEnumFilter(List<String> patterns) {
            super(patterns);
        }

        @Override
        public boolean isIgnored(String path) {
            return !super.isIgnored(path);
        }
    }

    public static void printUsage() {
        System.out.println("  java -jar gamepack.jar zpk pack [-k <key_name>] -i <input_dir> -o <output_dir>"
                + " -E <exclude_pattern> -E <exclude_pattern> ...");
        System.out.println("  java -jar gamepack.jar zpk pack [-k <key_name>] -i <input_dir> -o <output_dir>"
                + " -I <include_pattern> -I <include_pattern> ...");
    }

    public static void usage() {
        System.out.println("Usage:");
        printUsage();
        System.exit(1);
    }

    public static void callMain(String[] args) throws IOException {
        AbstractFile inputDir = null;
        AbstractFile outputFile = null;
        List<String> filterList = new ArrayList<>();
        int includeOrExclude = 0;
        String key = null;
        int i = args[1].equals("pack") ? 2 : 1;
        while (i < args.length) {
            switch (args[i++]) {
                case "-k" -> {
                    if (i < args.length) {
                        key = args[i++];
                    }
                }
                case "-i" -> {
                    if (i < args.length) {
                        inputDir = AbstractFile.make(args[i++]);
                    }
                }
                case "-o" -> {
                    if (i < args.length) {
                        outputFile = AbstractFile.make(args[i++]);
                    }
                }
                case "-E" -> {
                    if (i < args.length && includeOrExclude != 1) {
                        includeOrExclude = 2;
                        filterList.add(args[i++]);
                    }
                }
                case "-I" -> {
                    if (i < args.length) {
                        includeOrExclude = 1;
                        filterList.add(args[i++]);
                    }
                }
                case "-d" -> ENABLE_DEBUG = true;
                default -> usage();
            }
        }
        if (null == inputDir) {
            usage();
            return;
        } else if (!inputDir.isDirectory()) {
            System.err.println("File not a directory: " + inputDir);
            System.exit(2);
        }
        ZpkFilePack builder = new ZpkFilePack();
        final long buildTime = builder.buildTime;
        if (null == outputFile) {
            outputFile = AbstractFile.make(new File(".").getAbsolutePath());
        }
        if (isTextEmpty(key) && !outputFile.getName().toLowerCase().endsWith(EXT_NAME)) {
            key = inputDir.getName();
        }
        if (!isTextEmpty(key)) {
            outputFile = outputFile.getChildFile(key).getChildFile(getZpkFileName(key, buildTime));
        }
        System.out.println("Output File: " + outputFile);
        final long fileSize = builder.build(outputFile, inputDir,
                includeOrExclude == 1 ? new IncludesEnumFilter(filterList) :
                includeOrExclude == 2 ? new ExcludesEnumFilter(filterList) : null);
        final long cost = System.currentTimeMillis() - buildTime;
        System.out.println("Done! ");
        System.out.println("Output File: " + outputFile + ". "
                + "Size: " + humanReadableBytes(fileSize) + ". "
                + "Speed: " + humanReadableBytes(1000.0 * fileSize / cost) + "/s" + ". "
                + "Cost: " + humanReadableTime(cost));
        System.out.println();
        System.exit(0);
    }
}
